unit lissage;

interface

uses Math;

type TD2Coeff = array[0..2] of double; // 0 = degr 0, etc
     TLPoint = record
       X, Y: double;
     end;
     TLPList = array of TLPoint;
     TD2CList = array of TD2Coeff;

// Parabols
function GetD2Coeff(P1, P2, P3: TLPoint): TD2Coeff;
function GetParabolY(D2Coeff: TD2Coeff; x: double): double; overload;
// F: From, T: To, Pos: position
// Imagine you have a way from F to T, and you're at Pos from F. This is to set the "x" in the previous GetParabolY
// Pos = 0 means you're at F, Pos = 1 means you're at T, Pos = 0.5 means you're in the middle, and so on.
function GetParabolY(D2Coeff: TD2Coeff; F, T, Pos: double): double; overload;

// Curves
function GetCoeffList(Points: TLPList): TD2CList;
// Same as GetParabolY, but it needs the Point List so it can chose the right parabol to use
function GetCurveY(Coeffs: TD2CList; Points: TLPList; x: double): double; overload;
function GetCurveY(Coeffs: TD2CList; Points: TLPList; F, T, Pos: double): double; overload;
// The next function adds points in the middle of each arc of the curve and return a new lists of points
function GetSmoothCurvePoints(Coeffs: TD2CList; Points: TLPList): TLPList;

// Parametric parabols who can call a procedure to follow the path determined by three points
type TCallPathRunProc = procedure (X, Y: single; Data: Pointer);
     TCallPath = class
       Point1, Point2, Point3, Point4: TLPoint;
       constructor Create(P1, P2, P3, P4: TLPoint); virtual;
       procedure Run(FromT, ToT, StepT: single; Proc: TCallPathRunProc; Data: Pointer);
     end;

type TRandomCurve = class
       Heights: array of integer;
       // Warning: curve can go over max or under min because of the "parabol" maths.
       procedure Randomize(NbPoints, NbDetails, Min, Max: integer);
       constructor Create; virtual;
       destructor Destroy; override;
     end;

implementation

function GetD2Coeff(P1, P2, P3: TLPoint): TD2Coeff;
begin
  result[0]:=(sqr(P1.X)*(P2.X*P3.Y-P3.X*P2.Y)-P1.X*(sqr(P2.X)*P3.Y-sqr(P3.X)*P2.Y)+P2.X*(P2.X-P3.X)*P3.X*P1.Y)/((sqr(P1.X)-P1.X*(P2.X+P3.X)+P2.X*P3.X)*(P2.X-P3.X));
  result[1]:=(sqr(P1.X)*(P2.Y-P3.Y)-sqr(P2.X)*(P1.Y-P3.Y)+sqr(P3.X)*(P1.Y-P2.Y))/((sqr(P1.X)-P1.X*(P2.X+P3.X)+P2.X*P3.X)*(P2.X-P3.X));
  result[2]:=-(P1.X*(P2.Y-P3.Y)-P2.X*(P1.Y-P3.Y)+P3.X*(P1.Y-P2.Y))/((sqr(P1.X)-P1.X*(P2.X+P3.X)+P2.X*P3.X)*(P2.X-P3.X));
end;

function GetParabolY(D2Coeff: TD2Coeff; x: double): double;
begin
  result:=D2Coeff[2]*sqr(x)+D2Coeff[1]*x+D2Coeff[0];
end;

function GetParabolY(D2Coeff: TD2Coeff; F, T, Pos: double): double;
begin
  result:=GetParabolY(D2Coeff, F+(T-F)*Pos);
end;

function GetCoeffList(Points: TLPList): TD2CList;
var i: integer;
begin
  if Length(Points)<3 then
  begin
    SetLength(result, 0);
    exit;
  end;

  SetLength(result, Length(Points)-2);
  for i:=0 to Length(Points)-3 do
    result[i]:=GetD2Coeff(Points[i], Points[i+1], Points[i+2]);
end;

function GetCurveY(Coeffs: TD2CList; Points: TLPList; x: double): double; overload;
var L: integer;
    i, S: integer;
begin
  if Length(Coeffs)<>Length(Points)-2 then
  begin
    result:=0;
    exit;
  end;

  L:=Length(Coeffs)+1; // Segment count

  // Segment index
  if x<=Points[0].X then S:=1 else // First segment if outside (left)
  if x>=Points[length(Points)-1].X then S:=L else // Last segment if outside (right)
  begin
    S:=0;
    for i:=1 to L do
    begin
      if (x>=Points[i-1].X) and (x<=Points[i].X) then
      begin
        S:=i;
        break;
      end;
    end;
  end;

  // Finally, the Y value
  if (S<=0) or (S>L) then
  begin
    result:=0;
  end else begin
    // First segment
    if S=1 then
    begin
      result:=GetParabolY(Coeffs[0], x);
    end else
    // Last segment
    if S=L then
    begin
      result:=GetParabolY(Coeffs[L-2], x);
    end else
    // Other segments
    begin
      result:=(GetParabolY(Coeffs[S-2], x)+GetParabolY(Coeffs[S-1], x))/2;
      //result:=GetParabolY(Coeffs[S-2], x);
    end;
  end;
end;

function GetCurveY(Coeffs: TD2CList; Points: TLPList; F, T, Pos: double): double; overload;
begin
  result:=GetCurveY(Coeffs, Points, F+(T-F)*Pos);
end;

function GetSmoothCurvePoints(Coeffs: TD2CList; Points: TLPList): TLPList;
var i: integer;
begin
  SetLength(result, Length(Points)*2-1);
  for i:=0 to Length(Points)-2 do
  begin
    result[i*2]:=Points[i];
    result[i*2+1].X:=Points[i].X+(Points[i+1].X-Points[i].X)/2;
    result[i*2+1].Y:=GetCurveY(Coeffs, Points, result[i*2+1].X);
  end;
  result[length(Result)-1]:=Points[length(Points)-1];
end;

{ TCurve }

constructor TRandomCurve.Create;
begin
  SetLength(Heights, 0);
end;

destructor TRandomCurve.Destroy;
begin
  SetLength(Heights, 0);
end;

procedure TRandomCurve.Randomize(NbPoints, NbDetails, Min, Max: integer);
var P, SP: TLPList;
    C, SC: TD2CList;
    i: integer;
begin
  SetLength(P, NbPoints);
  for i:=0 to NbPoints-1 do
  begin
    P[i].X:=i/(NbPoints-1);
    P[i].Y:=random(Max-Min)+Min;
  end;
  C:=GetCoeffList(P);

  SP:=GetSmoothCurvePoints(C, P);
  SC:=GetCoeffList(SP);

  SetLength(Heights, NbDetails);
  for i:=0 to NbDetails-1 do
    Heights[i]:=round(GetCurveY(SC, SP, i/(NbDetails-1)));

  SetLength(P, 0);
  SetLength(SP, 0);
  SetLength(C, 0);
  SetLength(SC, 0);
end;

{ TCallParabol }

constructor TCallPath.Create(P1, P2, P3, P4: TLPoint);
begin
  Point1:=P1;
  Point2:=P2;
  Point3:=P3;
  Point4:=P4;
end;

procedure TCallPath.Run(FromT, ToT, StepT: single; Proc: TCallPathRunProc; Data: Pointer);
var t: single;
begin
  if (ToT-FromT)*StepT <= 0 then exit;

  t:=FromT;
  while t<ToT do
  begin
    t:=t+StepT;
    Proc( Power((1-t), 3)*Point1.X + 3*Power((1-t), 2)*t*Point2.X + 3*(1-t)*Power(t, 2)*Point3.X + Power(t, 3)*Point4.X,
          Power((1-t), 3)*Point1.Y + 3*Power((1-t), 2)*t*Point2.Y + 3*(1-t)*Power(t, 2)*Point3.Y + Power(t, 3)*Point4.Y,
          Data );
  end;
  //Bx(t) = (1-t)^3*P1x + 3*(1-t)^2*t*PC1x + 3*(1-t)*t^2 PC2x + t^3*P2x
  //By(t) = (1-t)^3*P1y + 3*(1-t)^2*t*PC1y + 3*(1-t)*t^2 PC2y + t^3*P2y
end;

end.
