unit W3files;

interface

uses Classes, SysUtils;

const // Main tilesets
      TS_Ashenvale = 'A';
      TS_Barrens = 'B';
      TS_Lordaeron_Fall = 'F';
      TS_Lordaeron_Summer = 'L';
      TS_Northrend = 'N';
      TS_Village = 'V';
      TS_Lordaeron_Winter = 'W';
      TS_City_Dalaran = 'X';
      TS_City_Lordaeron = 'Y';

      // Lordaeron Summer
      LSTILE_DIRT = 'Ldrt';
      LSTILE_ROUGH_DIRT = 'Ldro';
      LSTILE_GRASSY_DIRT = 'Ldrg';
      LSTILE_ROCK = 'Lrok';
      LSTILE_GRASS = 'Lgrs';
      LSTILE_DARK_GRASS = 'Lgrd';

      // Map flags (in w3i)
      // Melee maps usually use MFMelee_Map, MFMasked_Partially_Visible and MFAlways
      MFHide_Minimap_Preview = $0001;
      MFModify_Ally_Priorities = $0002;
      MFMelee_Map = $0004;
      MFLarge_Never_Reduced_To_Medium = $0008; // ?
      MFMasked_Partially_Visible = $0010;
      MFFixed_Player_Forces = $0020;
      MFCustom_Forces = $0040;
      MFCustom_Techtree = $0080;
      MFCustom_Abilities = $0100;
      MFCustom_Upgrades = $0200;
      MFProperties_Once_Opened = $0400; // ?
      MFShow_Waves_Cliff = $0800;
      MFShow_Waves_Rolling = $1000;
      MFAlways = $C000; // Always set (unused)

      // Loading screens and other kind of stuff like that
      SCNONE = $FFFFFFFF; // (-1)

      // Doodads names
      DOO_STARTLOCATION = 'sloc';
      DOO_GOLDMINE = 'ngol';
      DOO_GOBLIN_MERCHANT = 'ngme';
      DOO_HEALTH_FOUNTAIN = 'nfoh';
      DOO_MANA_FOUNTAIN = 'nmoo';
      DOO_GOBLIN_LAB = 'ngad';
      DOO_MERCENARIES = 'nmer';

      // Doodads flags
      DOOF_INVISIBLE_NONSOLID = 0;
      DOOF_VISIBLE_NONSOLID = 1;
      DOOF_VISIBLE_SOLID = 2;

      // Doodads other infos
      DOO_OWNER_PLAYER1 = 0;
      DOO_OWNER_PLAYER2 = 1;
      DOO_OWNER_PLAYER3 = 2;
      DOO_OWNER_PLAYER4 = 3;
      DOO_OWNER_PLAYER5 = 4;
      DOO_OWNER_PLAYER6 = 5;
      DOO_OWNER_PLAYER7 = 6;
      DOO_OWNER_PLAYER8 = 7;
      DOO_OWNER_PLAYER9 = 8;
      DOO_OWNER_PLAYER10 = 9;
      DOO_OWNER_PLAYER11 = 10;
      DOO_OWNER_PLAYER12 = 11;
      DOO_OWNER_NEUTRALPASSIVE = 16;

      DOO_HP_DEFAULT = -1;

      // Trees
      TREE_LORDAERON = 'LTlt';

type pTilePoint = ^TTilePoint;
     TTilePoint = record
       GroundHeight: Smallint; // Warning: 2 highest bits aren't used
       WaterLevel: Smallint;   // Warning: 2 highest bits aren't used
       Boundary1: boolean;     // Boundary (shadowed edges of the map)
       Ramp: boolean;          // Used to set a ramp between two "cliff levels" (layers)
       Blight: boolean;        // Undead's blight
       Water: boolean;         // Area filled by water. Water isn't shown if groundheight is too high.
       Boundary2: boolean;     // Used on "camera bounds" area
       TextureDetails: byte;   // Texture's details (holes, bumps...)
       GroundTile: 0..15;       // Index in TileSet
       CliffTexture: 0..15;     // Index in CliffSet
       LayerHeight: 0..15;      // "Cliff level" (layer)
     end;
     TMap_w3e = class // Environment
     private
       FMain_Tileset: char;
       FMW, FMH: integer;
       function SaveHeaderToStream(Stream: TStream): integer;
       function SaveDataToStream(Stream: TStream): integer;
       procedure SetMapWidth(value: integer);
       procedure SetMapHeight(value: integer);
     public
       Map: array of array of TTilePoint;
       TileSet, CliffSet: TStringList; // Never put more than 16 tiles in a tileset
       function SaveToStream(Stream: TStream): integer;
       property Main_Tileset: char read FMain_Tileset write FMain_Tileset;
       property MapWidth: integer read FMW write SetMapWidth;
       property MapHeight: integer read FMH write SetMapHeight;
       constructor Create; virtual;
       destructor Destroy; override;
     end;

     pPlayerInfo = ^TPlayerInfo;
     TPlayerInfo = record
       FixedPosition: boolean;
       StartingX, StartingY: single;
       LowPriorFlags, HighPriorFlags: cardinal; // bit x set means "set for player x"
     end;

     TMap_w3i = class // Map infos
     private
       FMain_Tileset: char;
     public
       Players: array of pPlayerInfo;
       NbForces: integer;
       procedure ClearPlayers;
       function SaveToStream(Stream: TStream): integer;
       property Main_Tileset: char read FMain_Tileset write FMain_Tileset;
       constructor Create; virtual;
       destructor Destroy; override;
     end;

     pDoodad = ^TDoodad;
     TDoodad = record
       Dootype: array[1..4] of char;
       X, Y, Z: single;
       Rotation: single;
       ScaleX, ScaleY, ScaleZ: single;
       Flags: byte;
       Owner: cardinal;
       HP: integer;
       Gold: integer; // for gold mines
     end;

     TMap_UnitsDoo = class // Doodads units (including start locations)
       Players: array of pPlayerInfo;
       Others: array of pDoodad;
       procedure ClearPlayers;
       function SaveToStream(Stream: TStream): integer;
       constructor Create; virtual;
       destructor Destroy; override;
       function AddGoldMine(PosX, PosY, PosZ: single; GoldCount: integer): pDoodad;
     end;

     // Warning - this wts is hard to use as it refers to other files. Use Tw3Map instead.
     // Note: I chose to start TRIGSTR indexes at 1
     TMap_wts = class // TRIGSTR_xxx strings
       strings: TStringList;
       function SaveToStream(Stream: TStream): integer;
       constructor Create; virtual;
       destructor Destroy; override;
     end;

     pTree = ^TTree;
     TTree = record
       X, Y, Z: single;
       Rotation: single; // (Radian)
       ScaleX, ScaleY, ScaleZ: single;
       ID: array[1..4] of char;
       Variation: integer;
       Life: byte; // % of life
     end;

     TMap_TreeDoo = class
     private
       procedure SaveTree(Stream: TStream; Tree: pTree; Index: integer);
     public
       Trees: array of pTree;
       function AddTree: pTree;
       function SaveToStream(Stream: TStream): integer;
       constructor Create; virtual;
       destructor Destroy; override;
       procedure ClearTrees;
     end;

     Tw3Map = class // All files (not packed); using defaults melee parameters
     private
       FMain_TileSet: char;
       Trees: array of array of boolean; // array[Y, X], true means there is a tree
       CADTable: array of array of boolean; // array[Y, X], true means you can add a doodad here
       procedure SetMain_TileSet(value: char);
       procedure UpdateDDTrees;
       function ThereIsASloc(X, Y: integer): boolean;
       procedure Block(X, Y: single; Size: integer);
     public
       w3e: TMap_w3e;
       w3i: TMap_w3i;
       DestructibleDoo: TMap_TreeDoo;
       UnitsDoo: TMap_UnitsDoo;
       wts: TMap_wts;
       procedure ClearPlayers;
       function AddPlayer: pPlayerInfo;
       function Player(index: integer): pPlayerInfo;
       constructor Create(Width, Height: byte); virtual;
       destructor Destroy; override;
       property Main_Tileset: char read FMain_Tileset write SetMain_Tileset;
       procedure RebuildWts;
       procedure GenerateFiles(Path: string);
       function AddGoldMine(PosX, PosY: single; GoldCount: integer): pDoodad;
       // GetTilePoint returns the TilePoint at PosX, PosY (using: size of a tile: 128; center of the map: 0, 0)
       function GetTilePoint(PosX, PosY: single): pTilePoint;
       procedure SetTree(X, Y: integer; Value: boolean);
       function GetTree(X, Y: integer): boolean;
       function CanAddDoodad(X, Y: integer): boolean;
       function IsValidCoord(X, Y: integer): boolean;
       function FCoordToICoordX(X: single): integer;
       function FCoordToICoordY(Y: single): integer;
       function ICoordToFCoordX(X: integer): single;
       function ICoordToFCoordY(Y: integer): single;
     end;

implementation

procedure WriteW3String(Stream: TStream; Str: string);
var i: integer;
    c: char;
begin
  for i:=1 to length(Str) do
  begin
    c:=Str[i];
    Stream.Write(c, 1);
  end;
  c:=#0;
  Stream.Write(c, 1);
end;

procedure WriteLine(Stream: TStream; Str: string);
var i: integer;
    c: char;
    b: byte;
begin
  for i:=1 to length(Str) do
  begin
    c:=Str[i];
    Stream.Write(c, 1);
  end;
  b:=$0D; Stream.Write(b, 1);
  b:=$0A; Stream.Write(b, 1);
end;

var nextint: integer;

procedure FirstInt;
begin
  nextint:=0;
end;

function GetNextInt(Size: byte): string;
begin
  inc(nextint);
  result:=inttostr(nextint);
  while length(result)<Size do result:='0'+result;
end;

{ TMap_w3e }

constructor TMap_w3e.Create;
begin
  TileSet:=TStringList.Create;
  CliffSet:=TStringList.Create;
  SetLength(Map, 0);
end;

destructor TMap_w3e.Destroy;
begin
  TileSet.Free;
  CliffSet.Free;
  inherited Destroy;
end;

function TMap_w3e.SaveDataToStream(Stream: TStream): integer;
var OldPos: integer;
    x, y: integer;
    TilePoint: TTilePoint;
    si: smallint;
    b: byte;
begin
  OldPos:=Stream.Position;

  {Each tilepoint is defined by a block of 7 bytes.
The number of blocks is equal to Mx*My.
short: ground height
C000h: minimum height (-16384)
2000h: normal height (ground level 0)
3FFFh: maximum height (+16383)
short: water level + map edge boundary flag*(see notes)
4bit: flags*(see notes)
4bit: ground texture type (grass, dirt, rocks,...)
1byte: texture details (rocks, holes, bones,...)
4bit: cliff texture type
4bit: layer height

*flags notes:
Flags values:
0x4000 --> boundary flag 1 (shadow generated by the world editor on the edge of the map)
0x0010 --> ramp flag (used to set a ramp between two layers)
0x0020 --> blight flag (ground will look like Undead's ground)
0x0040 --> water flag (enable water)
0x0080 --> boundary flag 2 (used on "camera bounds" area. Usually set by the World Editor "boundary" tool.)

Water level:
Water level is stored just like ground height. The highest bit (bit 15) is used for the boundary flag 1.}

  for y:=FMH downto 0 do
  begin
    for x:=0 to FMW do
    begin
      TilePoint:=Map[y, x];

      // 2 bytes
      si:=TilePoint.GroundHeight and $3FFF;
      Stream.Write(si, 2);

      // 2 bytes
      si:=TilePoint.WaterLevel and $3FFF;
      if TilePoint.Boundary1 then si:=si or $4000;
      Stream.Write(si, 2);

      // 1 byte
      b:=TilePoint.GroundTile;
      if TilePoint.Ramp then b:=b or $10;
      if TilePoint.Blight then b:=b or $20;
      if TilePoint.Water then b:=b or $40;
      if TilePoint.Boundary2 then b:=b or $80;
      Stream.Write(b, 1);

      // 1 byte
      b:=TilePoint.TextureDetails;
      Stream.Write(b, 1);

      // 1 byte
      b:=(TilePoint.CliffTexture shl 4) or TilePoint.LayerHeight;
      Stream.Write(b, 1);
    end;
  end;

  result:=Stream.Position-OldPos;
end;

function TMap_w3e.SaveHeaderToStream(Stream: TStream): integer;
var b: byte;
    c: cardinal;
    OldPos: integer;
    i, n: integer;
    s: string;
    f: Single;
    ch: char;
begin
  OldPos:=Stream.Position;

  // file format
  Stream.Write('W3E!', 4);
  // file version
  b:=$0B; Stream.Write(b, 1); b:=$00; Stream.Write(b, 1); b:=$00; Stream.Write(b, 1); b:=$00; Stream.Write(b, 1);
  // tileset
  Stream.Write(Main_Tileset, 1);
  c:=0; Stream.Write(c, 4); // not using custom tilesets
  c:=TileSet.Count; Stream.Write(c, 4); // Number of tilesets used
  for i:=0 to TileSet.Count-1 do
  begin
    s:=TileSet[i];// Stream.Write(s, 4);
    while length(s)<4 do s:=s+' ';
    for n:=1 to 4 do
    begin
      ch:=s[n];
      Stream.Write(ch, 1);
    end;
  end;
  c:=CliffSet.Count; Stream.Write(c, 4); // Number of cliff tilesets used
  for i:=0 to CliffSet.Count-1 do
  begin
    s:=CliffSet[i];// Stream.Write(s, 4);
    while length(s)<4 do s:=s+' ';
    for n:=1 to 4 do
    begin
      ch:=s[n];
      Stream.Write(ch, 1);
    end;
  end;
  c:=MapWidth+1; Stream.Write(c, 4);
  c:=MapHeight+1; Stream.Write(c, 4);
  f:=-1*MapWidth*128/2; Stream.Write(f, 4);
  f:=-1*MapHeight*128/2; Stream.Write(f, 4);

  result:=Stream.Position-OldPos;
end;

function TMap_w3e.SaveToStream(Stream: TStream): integer;
begin
  result:=0;
  result:=result+SaveHeaderToStream(Stream);
  result:=result+SaveDataToStream(Stream);
end;

procedure TMap_w3e.SetMapHeight(value: integer);
var old, y: integer;
begin
  if value>256 then value:=256;
  if value<32 then value:=32;
  FMH:=value;
  old:=Length(Map);
  SetLength(Map, FMH+1);
  for y:=old to length(Map)-1 do SetLength(Map[y], FMW+1);
end;

procedure TMap_w3e.SetMapWidth(value: integer);
var y: integer;
begin
  if value>256 then value:=256;
  if value<32 then value:=32;
  FMW:=value;
  for y:=0 to length(Map)-1 do SetLength(Map[y], FMW+1);
end;

{ TMap_w3i }

procedure TMap_w3i.ClearPlayers;
var i: integer;
begin
  for i:=0 to length(Players)-1 do dispose(Players[i]);
  SetLength(Players, 0);
end;

constructor TMap_w3i.Create;
begin
  SetLength(Players, 0);
  NbForces:=0;
end;

destructor TMap_w3i.Destroy;
begin
  ClearPlayers;
  inherited Destroy;
end;

function TMap_w3i.SaveToStream(Stream: TStream): integer;
var b: byte;
    c: cardinal;
    OldPos: integer;
    i: integer;
    PI: pPlayerInfo;
begin
  FirstInt;

  OldPos:=Stream.Position;

  // File format kind of things
  c:=18; Stream.Write(c, 4); // File format version
  c:=1; Stream.Write(c, 4); // Number of saves (map version)
  c:=0; Stream.Write(c, 4); // Editor version
  // Map descriptions
  WriteW3String(Stream, 'Random map'); // Title
  WriteW3String(Stream, 'DooMeeR''s W3MGen'); // Author
  WriteW3String(Stream, 'Random map generated by DooMeeR''s W3MGen.'); // Description
  WriteW3String(Stream, '2'); // Players
  // Don't know the signification of the following floats. They are map boundaries, but...
  c:=$c5300000; Stream.Write(c, 4);
  c:=$c5500000; Stream.Write(c, 4);
  c:=$45300000; Stream.Write(c, 4);
  c:=$45300000; Stream.Write(c, 4);
  c:=$c5300000; Stream.Write(c, 4);
  c:=$45300000; Stream.Write(c, 4);
  c:=$45300000; Stream.Write(c, 4);
  c:=$c5500000; Stream.Write(c, 4);
  // Boundaries (left, right, down (near 0), up (near maxheight))
  c:=6; Stream.Write(c, 4);
  c:=6; Stream.Write(c, 4);
  c:=4; Stream.Write(c, 4);
  c:=8; Stream.Write(c, 4);
  // Playable area (width, height)
  c:=52; Stream.Write(c, 4);
  c:=52; Stream.Write(c, 4);
  // Flags
  c:=MFMelee_Map or MFMasked_Partially_Visible or MFAlways or MFProperties_Once_Opened; Stream.Write(c, 4);
  // Main tileset
  Stream.Write(Main_Tileset, 1);
  // Campaign background number
  c:=SCNONE; Stream.Write(c, 4);
  // Loading screen text, title and subtitle (here empty; they are just strings)
  b:=0; Stream.Write(b, 1);
  b:=0; Stream.Write(b, 1);
  b:=0; Stream.Write(b, 1);
  // Map loading screen number
  c:=SCNONE; Stream.Write(c, 4);
  // Prologue screen text, title and subtitle (here empty; they are just strings)
  b:=0; Stream.Write(b, 1);
  b:=0; Stream.Write(b, 1);
  b:=0; Stream.Write(b, 1);
  // Max number of players
  c:=length(Players); Stream.Write(c, 4);
  // Players
  for i:=0 to length(Players)-1 do
  begin
    PI:=Players[i];
    c:=i; Stream.Write(c, 4); // ?
    c:=1; Stream.Write(c, 4); // 1=Human, 2=Computer, 3=Neutral, 4=Rescuable
    c:=1; Stream.Write(c, 4); // 1=Human, 2=Orc, 3=Undead, 4=Night Elf
    if PI^.FixedPosition then c:=$01000000 else c:=0; Stream.Write(c, 4);
    WriteW3String(Stream, 'TRIGSTR_'+GetNextInt(3)); // Player name...
    Stream.Write(PI^.StartingX, 4);
    Stream.Write(PI^.StartingY, 4);
    Stream.Write(PI^.LowPriorFlags, 4);
    Stream.Write(PI^.HighPriorFlags, 4);
  end;
  // Forces
  Stream.Write(NbForces, 4); // Max number of forces
  c:=0; Stream.Write(c, 4); // Flags (melee => no flag set)
  c:=$FFFFFFFF; Stream.Write(c, 4); // unknown
  for i:=1 to NbForces do
  begin
    WriteW3String(Stream, 'TRIGSTR_'+GetNextInt(3)); // Force name...
    c:=0; Stream.Write(c, 4); // unknown
    c:=0; Stream.Write(c, 4); // unknown
  end;
  // End
  c:=0; Stream.Write(c, 4); // end of w3i

  result:=Stream.Position-OldPos;
end;

{ Tw3Map }

function Tw3Map.AddGoldMine(PosX, PosY: single;
  GoldCount: integer): pDoodad;
begin
  PosX:=trunc(PosX/64)*64;
  PosY:=trunc(PosY/64)*64;
  result:=UnitsDoo.AddGoldMine(PosX, PosY, GetTilePoint(PosX, PosY)^.GroundHeight-$2000, GoldCount);

  Block(PosX, PosY, 2);
end;

function Tw3Map.AddPlayer: pPlayerInfo;
begin
  SetLength(w3i.Players, length(w3i.Players)+1);
  SetLength(UnitsDoo.Players, length(UnitsDoo.Players)+1);
  new(w3i.Players[length(w3i.Players)-1]);
  result:=w3i.Players[length(w3i.Players)-1];
  UnitsDoo.Players[length(UnitsDoo.Players)-1]:=result;

  RebuildWts;
end;

procedure Tw3Map.ClearPlayers;
begin
  w3i.ClearPlayers;
  SetLength(UnitsDoo.Players, 0);

  RebuildWts;
end;

constructor Tw3Map.Create(Width, Height: byte);
var i, n: integer;
begin
  w3e:=TMap_w3e.Create;
  w3e.MapWidth:=Width;
  w3e.MapHeight:=Height;
  w3i:=TMap_w3i.Create;
  UnitsDoo:=TMap_UnitsDoo.Create;
  wts:=TMap_wts.Create;
  SetLength(Trees, Height);
  SetLength(CADTable, Height);
  for i:=0 to Height-1 do
  begin
    SetLength(Trees[i], Width);
    SetLength(CADTable[i], Width);
    for n:=0 to Width-1 do
    begin
      Trees[i, n]:=false;
      CADTable[i, n]:=true;
    end;
  end;
  DestructibleDoo:=TMap_TreeDoo.Create;

  RebuildWts;
end;

destructor Tw3Map.Destroy;
var i: integer;
begin
  ClearPlayers;
  w3e.Free;
  w3i.Free;
  UnitsDoo.Free;
  wts.Free;
  for i:=0 to length(Trees)-1 do SetLength(Trees[i], 0);
  SetLength(Trees, 0);
  DestructibleDoo.Free;
  inherited Destroy;
end;

procedure Tw3Map.GenerateFiles(Path: string);
var F: TFileStream;
begin
  if Path<>'' then if Path[length(Path)]<>'\' then Path:=Path+'\';

  F:=TFileStream.Create(Path+'war3map.w3e', fmCreate or fmShareExclusive);
  w3e.SaveToStream(F);
  F.Free;

  F:=TFileStream.Create(Path+'war3map.w3i', fmCreate or fmShareExclusive);
  w3i.SaveToStream(F);
  F.Free;

  F:=TFileStream.Create(Path+'war3mapUnits.doo', fmCreate or fmShareExclusive);
  UnitsDoo.SaveToStream(F);
  F.Free;

  F:=TFileStream.Create(Path+'war3map.wts', fmCreate or fmShareExclusive);
  wts.SaveToStream(F);
  F.Free;

  UpdateDDTrees;
  F:=TFileStream.Create(Path+'war3map.doo', fmCreate or fmShareExclusive);
  DestructibleDoo.SaveToStream(F);
  F.Free;
end;

function Tw3Map.GetTilePoint(PosX, PosY: single): pTilePoint;
var X, Y: integer;
begin
  X:=(round(PosX) div 128) + (w3e.MapWidth div 2);
  Y:=(round(PosY) div 128) + (w3e.MapHeight div 2);
  if (X>=0) and (Y>=0) and (X<=w3e.MapWidth) and (Y<=w3e.MapHeight) then
    result:=@w3e.Map[Y, X] else result:=nil;
end;

function Tw3Map.GetTree(X, Y: integer): boolean;
begin
  if IsValidCoord(X, Y) then
    result:=Trees[Y, X] else result:=false;
end;

procedure Tw3Map.SetTree(X, Y: integer; Value: boolean);
begin
  if IsValidCoord(X, Y) then
  begin
    if Value=true then
    begin
      if CanAddDoodad(X, Y) then
      begin
        Trees[Y, X]:=true;
        CADTable[Y, X]:=false;
      end;
    end else begin
      if Trees[Y, X] then
      begin
        Trees[Y, X]:=false;
        CADTable[Y, X]:=true;
      end;
    end;
  end;
end;

function Tw3Map.Player(index: integer): pPlayerInfo;
begin
  if (index>=0) and (index<length(w3i.Players)) then
    result:=w3i.Players[index]
  else result:=nil;
end;

procedure Tw3Map.RebuildWts;
var i: integer;
begin
  wts.strings.Clear;
  for i:=0 to length(w3i.Players)-1 do wts.strings.Add('Player '+inttostr(i+1));
  wts.strings.Add('Force 1');
end;

procedure Tw3Map.SetMain_TileSet(value: char);
begin
  FMain_TileSet:=value;
  w3e.Main_Tileset:=FMain_TileSet;
  w3i.Main_Tileset:=FMain_TileSet;
end;

procedure Tw3Map.UpdateDDTrees;
var x, y: integer;
    Tree: pTree;
begin
  DestructibleDoo.ClearTrees;
  for y:=0 to Length(Trees)-1 do
    for x:=0 to Length(Trees[y])-1 do
      if Trees[y, x] then
      begin
        Tree:=DestructibleDoo.AddTree;
        Tree^.X:=(x-Length(Trees[y]) div 2)*128+64;
        Tree^.Y:=(y-Length(Trees) div 2)*128+64;
        Tree^.Z:=w3e.Map[y, x].GroundHeight-$2000;
      end;
end;

function Tw3Map.CanAddDoodad(X, Y: integer): boolean;
begin
  if IsValidCoord(X, Y) then result:=CADTable[Y, X] or ThereIsASloc(X, Y) else result:=false;
end;

function Tw3Map.IsValidCoord(X, Y: integer): boolean;
begin
  result:=(X>=0) and (Y>=0) and (X<w3e.MapWidth) and (Y<w3e.MapHeight);
end;

function Tw3Map.ThereIsASloc(X, Y: integer): boolean;
var i: integer;
    BX, BY: integer;
begin
  result:=false;
  for i:=0 to Length(UnitsDoo.Players)-1 do
  begin
    BX:=FCoordToICoordX(UnitsDoo.Players[i]^.StartingX);
    BY:=FCoordToICoordY(UnitsDoo.Players[i]^.StartingY);
    if (X>=BX-2) and (X<=BX+1) and (Y>=BY-2) and (Y<=BY+1) then result:=true;
  end;
end;

function Tw3Map.FCoordToICoordX(X: single): integer;
begin
  result:=trunc(X/128)+w3e.MapWidth div 2;
end;

function Tw3Map.FCoordToICoordY(Y: single): integer;
begin
  result:=trunc(Y/128)+w3e.MapHeight div 2;
end;

function Tw3Map.ICoordToFCoordX(X: integer): single;
begin
  result:=(X-w3e.MapWidth)*128;
end;

function Tw3Map.ICoordToFCoordY(Y: integer): single;
begin
  result:=(Y-w3e.MapHeight)*128;
end;

procedure Tw3Map.Block(X, Y: single; Size: integer);
var iX, iY: integer;
begin
  X:=X+128*w3e.MapWidth/2;
  Y:=w3e.MapWidth*128-(Y+128*w3e.MapWidth/2);
  for iX:=0 to w3e.MapWidth-1 do
    for iY:=0 to w3e.MapHeight-1 do
    begin
      if (abs(iX*128+64 - X) <= Size*128) and (abs(iY*128+64 - Y) <= Size*128) then
        if IsValidCoord(iX, iY) then CADTable[iY, iX]:=false;
    end;
end;

{ TMap_UnitsDoo }

function TMap_UnitsDoo.AddGoldMine(PosX, PosY, PosZ: single;
  GoldCount: integer): pDoodad;
begin
  SetLength(Others, Length(Others)+1);
  new(result);
  Others[length(Others)-1]:=result;
  with result^ do
  begin
    Dootype:=DOO_GOLDMINE;
    X:=PosX;
    Y:=PosY;
    Z:=PosZ;
    Rotation:=0;
    ScaleX:=1;
    ScaleY:=1;
    ScaleZ:=1;
    Flags:=DOOF_VISIBLE_SOLID;
    Owner:=DOO_OWNER_NEUTRALPASSIVE;
    HP:=0;
    Gold:=GoldCount;
  end;
end;

procedure TMap_UnitsDoo.ClearPlayers;
var i: integer;
begin
  for i:=0 to length(Players)-1 do dispose(Players[i]);
  SetLength(Players, 0);
end;

constructor TMap_UnitsDoo.Create;
begin
  SetLength(Players, 0);
  SetLength(Others, 0);
end;

destructor TMap_UnitsDoo.Destroy;
begin
  ClearPlayers;
  inherited Destroy;
end;

function TMap_UnitsDoo.SaveToStream(Stream: TStream): integer;
var OldPos: integer;
    c: cardinal;
    i, n: integer;
    f: single;
    b: byte;
    Doodad: pDoodad;
begin
  OldPos:=Stream.Position;

  // Header
  Stream.Write('W3do', 4);
  c:=7; Stream.Write(c, 4);
  c:=9; Stream.Write(c, 4);
  c:=length(Players)+length(Others); Stream.Write(c, 4);

  // Doodads
  // Player start locations
  for i:=0 to Length(Players)-1 do
  begin
    Stream.Write(DOO_STARTLOCATION, 4);
    c:=0; Stream.Write(c, 4); // Variation
    f:=Players[i]^.StartingX; Stream.Write(f, 4); // X
    f:=Players[i]^.StartingY; Stream.Write(f, 4); // Y
    f:=0; Stream.Write(f, 4); // Z
    f:=0; Stream.Write(f, 4); // Rotation angle, I've seen $4096CBE5 somewhere... dunno if it matters for a start location
    c:=$3F800000; Stream.Write(c, 4); // Scale X (float, here = 1)
    c:=$3F800000; Stream.Write(c, 4); // Scale Y (float, here = 1)
    c:=$3F800000; Stream.Write(c, 4); // Scale Z (float, here = 1)
    b:=DOOF_VISIBLE_SOLID; Stream.Write(b, 1); // Flags
    c:=i; Stream.Write(c, 4); // Owner (here: player num starting from 0)
    b:=0; Stream.Write(b, 1); b:=0; Stream.Write(b, 1); // Unknown
    c:=0; Stream.Write(c, 4); // Hit Points
    c:=0; Stream.Write(c, 4); // Unknown, usually -1 but I've seen 0 for sloc
    c:=0; Stream.Write(c, 4); // No dropped item sets
    c:=0; Stream.Write(c, 4); // No gold
    c:=0; Stream.Write(c, 4); // Unknown float often -1 but I've seen 0 for sloc
    c:=0; Stream.Write(c, 4); // Unknown int often -1 but I've seen 0 for sloc
    c:=0; Stream.Write(c, 4); // Unknown int I've seen 0 for sloc
    c:=0; Stream.Write(c, 4); // Unknown int I've seen 0 for sloc
    c:=0; Stream.Write(c, 4); // Random unit: Any neutral passive building (?)
    c:=0; Stream.Write(c, 4); // Random unit type (?)
    c:=$FFFFFFFF; Stream.Write(c, 4); // Custom color (-1 = none)
    c:=$FFFFFFFF; Stream.Write(c, 4); // No active destination number
    c:=i; Stream.Write(c, 4); // Creation number (?)
  end;

  // Other doodads
  for i:=0 to Length(Others)-1 do
  begin
    Doodad:=Others[i];
    Stream.Write(Doodad^.Dootype, 4);
    c:=0; Stream.Write(c, 4); // Variation
    f:=Doodad^.X; Stream.Write(f, 4); // X
    f:=Doodad^.Y; Stream.Write(f, 4); // Y
    f:=Doodad^.Z; Stream.Write(f, 4); // Z
    f:=Doodad^.Rotation; Stream.Write(f, 4); // Rotation angle
    f:=Doodad^.ScaleX; Stream.Write(f, 4); // Scale X
    f:=Doodad^.ScaleY; Stream.Write(f, 4); // Scale Y
    f:=Doodad^.ScaleZ; Stream.Write(f, 4); // Scale Z
    b:=Doodad^.Flags; Stream.Write(b, 1); // Flags
    c:=Doodad^.Owner; Stream.Write(c, 4); // Owner (here: player num starting from 0)
    b:=0; Stream.Write(b, 1); b:=0; Stream.Write(b, 1); // Unknown
    c:=Doodad^.HP; Stream.Write(c, 4); // Hit Points
    n:=0; Stream.Write(n, 4); // Unknown, usually -1
    c:=0; Stream.Write(c, 4); // No dropped item sets
    c:=Doodad^.Gold; Stream.Write(c, 4); // No gold
    f:=0; Stream.Write(f, 4); // Unknown float often -1
    n:=0; Stream.Write(n, 4); // Unknown int often -1
    c:=0; Stream.Write(c, 4); // Unknown int
    c:=0; Stream.Write(c, 4); // Unknown int
    c:=0; Stream.Write(c, 4); // Random unit: Any neutral passive building (?)
    c:=0; Stream.Write(c, 4); // Random unit type (?)
    c:=$FFFFFFFF; Stream.Write(c, 4); // Custom color (-1 = none)
    c:=$FFFFFFFF; Stream.Write(c, 4); // No active destination number
    c:=i+Length(Players); Stream.Write(c, 4); // Creation number (?)
  end;

  result:=Stream.Position-OldPos;
end;

{ TMap_wts }

constructor TMap_wts.Create;
begin
  Strings:=TStringList.Create;
end;

destructor TMap_wts.Destroy;
begin
  Strings.Free;
  inherited Destroy;
end;

function TMap_wts.SaveToStream(Stream: TStream): integer;
var OldPos, i: integer;
    b: byte;
begin
  OldPos:=Stream.Position;

  // It's a text file but I've seen 3 weird bytes at the beginning...
  b:=$EF; Stream.Write(b, 1);
  b:=$BB; Stream.Write(b, 1);
  b:=$BF; Stream.Write(b, 1);

  // Now the strings
  for i:=0 to strings.Count-1 do
  begin
    WriteLine(Stream, 'STRING '+inttostr(i+1));
    WriteLine(Stream, '{');
    WriteLine(Stream, strings[i]);
    WriteLine(Stream, '}');
  end;

  result:=Stream.Position-OldPos;
end;

{ TMap_TreeDoo }

function TMap_TreeDoo.AddTree: pTree;
begin
  SetLength(Trees, Length(Trees)+1);
  new(Trees[Length(Trees)-1]);
  result:=Trees[Length(Trees)-1];
  result.X:=0;
  result.Y:=0;
  result.Z:=0;
  result.Rotation:=0;
  result.ScaleX:=(random(40)+80)/100;
  result.ScaleY:=result.ScaleX;
  result.ScaleZ:=result.ScaleX;
  result.ID:=TREE_LORDAERON;
  result.Variation:=random(16);
  result.Life:=100;
end;

procedure TMap_TreeDoo.ClearTrees;
var i: integer;
begin
  for i:=0 to length(Trees)-1 do
    dispose(Trees[i]);
  SetLength(Trees, 0);
end;

constructor TMap_TreeDoo.Create;
begin
  SetLength(Trees, 0);
end;

destructor TMap_TreeDoo.Destroy;
begin
  ClearTrees;
  inherited Destroy;
end;

function TMap_TreeDoo.SaveToStream(Stream: TStream): integer;
var OldPos, i: integer;
begin
  OldPos:=Stream.Position;

  // Header
  Stream.Write('W3do', 4);
  i:=7; Stream.Write(i, 4);
  i:=9; Stream.Write(i, 4);
  i:=Length(Trees); Stream.Write(i, 4); // Tree count

  // Trees
  for i:=0 to Length(Trees)-1 do
    SaveTree(Stream, Trees[i], i);

  // End
  i:=0; Stream.Write(i, 4);
  i:=0; Stream.Write(i, 4);

  result:=Stream.Position-OldPos;
end;

procedure TMap_TreeDoo.SaveTree(Stream: TStream; Tree: pTree; Index: integer);
var b: byte;
begin
{char[4]: Tree ID (can be found in the file "Units\DestructableData.slk")
int: Variation (little endian)
float: Tree X coordinate on the map
float: Tree Y coordinate on the map
float: Tree Z coordinate on the map
float: Tree angle (radian angle value)(degree = radian*180/pi)
float: Tree X scale
float: Tree Y scale
float: Tree Z scale
byte: Tree flags*
byte: Tree life (integer stored in %, 100% is 0x64, 170% is 0xAA for example)
int: Tree ID number in the World Editor (little endian) (each tree has a different one)

*flags:
0= invisible and non-solid tree
1= visible but non-solid tree
2= normal tree (visible and solid)}
{     TTree = record
       X, Y, Z: single;
       Rotation: single; // (Radian)
       ScaleX, ScaleY, ScaleZ: single;
       ID: array[1..4] of char;
       Variation: integer;
       Life: byte; // % of life
     end;
}
  with Tree^ do
  begin
    Stream.Write(ID, 4);
    Stream.Write(Variation, 4);
    Stream.Write(X, 4);
    Stream.Write(Y, 4);
    Stream.Write(Z, 4);
    Stream.Write(Rotation, 4);
    Stream.Write(ScaleX, 4);
    Stream.Write(ScaleY, 4);
    Stream.Write(ScaleZ, 4);
    b:=DOOF_VISIBLE_SOLID; Stream.Write(b, 1);
    Stream.Write(Life, 1);
    Stream.Write(Index, 4);
  end;
end;

end.
